/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.compaction;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.inject.Inject;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.druid.client.indexing.ClientCompactionTaskQuery;
import org.apache.druid.indexer.TaskState;
import org.apache.druid.indexer.TaskStatus;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.server.compaction.CompactionCandidate;
import org.apache.druid.server.compaction.CompactionCandidateSearchPolicy;
import org.apache.druid.server.compaction.CompactionStatus;
import org.apache.druid.server.compaction.CompactionTaskStatus;
import org.apache.druid.server.coordinator.DataSourceCompactionConfig;
import org.apache.druid.server.coordinator.DruidCompactionConfig;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Interval;
import org.joda.time.ReadableDuration;
import org.joda.time.ReadableInstant;

public class CompactionStatusTracker {
    private static final Duration MAX_STATUS_RETAIN_DURATION = Duration.standardHours((long)12L);
    private final ObjectMapper objectMapper;
    private final ConcurrentHashMap<String, DatasourceStatus> datasourceStatuses = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, CompactionCandidate> submittedTaskIdToSegments = new ConcurrentHashMap();
    private final AtomicReference<DateTime> segmentSnapshotTime = new AtomicReference();

    @Inject
    public CompactionStatusTracker(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    public void stop() {
        this.datasourceStatuses.clear();
    }

    public void removeDatasource(String datasource) {
        this.datasourceStatuses.remove(datasource);
    }

    public CompactionTaskStatus getLatestTaskStatus(CompactionCandidate candidates) {
        return this.datasourceStatuses.getOrDefault((Object)candidates.getDataSource(), (DatasourceStatus)DatasourceStatus.EMPTY).intervalToTaskStatus.get(candidates.getUmbrellaInterval());
    }

    public Set<String> getSubmittedTaskIds() {
        return this.submittedTaskIdToSegments.keySet();
    }

    public CompactionStatus computeCompactionStatus(CompactionCandidate candidate, DataSourceCompactionConfig config, CompactionCandidateSearchPolicy searchPolicy) {
        CompactionStatus compactionStatus = CompactionStatus.compute(candidate, config, this.objectMapper);
        if (compactionStatus.isComplete()) {
            return compactionStatus;
        }
        long inputSegmentSize = config.getInputSegmentSizeBytes();
        if (candidate.getTotalBytes() > inputSegmentSize) {
            return CompactionStatus.skipped("'inputSegmentSize' exceeded: Total segment size[%d] is larger than allowed inputSegmentSize[%d]", candidate.getTotalBytes(), inputSegmentSize);
        }
        CompactionTaskStatus lastTaskStatus = this.getLatestTaskStatus(candidate);
        if (lastTaskStatus != null && lastTaskStatus.getState() == TaskState.RUNNING) {
            return CompactionStatus.skipped("Task for interval is already running", new Object[0]);
        }
        DateTime snapshotTime = this.segmentSnapshotTime.get();
        if (lastTaskStatus != null && lastTaskStatus.getState() == TaskState.SUCCESS && snapshotTime != null && snapshotTime.isBefore((ReadableInstant)lastTaskStatus.getUpdatedTime())) {
            return CompactionStatus.skipped("Segment timeline not updated since last compaction task succeeded", new Object[0]);
        }
        if (!searchPolicy.isEligibleForCompaction(candidate, compactionStatus, lastTaskStatus)) {
            return CompactionStatus.skipped("Rejected by search policy", new Object[0]);
        }
        return compactionStatus;
    }

    public void onCompactionStatusComputed(CompactionCandidate candidateSegments, DataSourceCompactionConfig config) {
    }

    public void onSegmentTimelineUpdated(DateTime snapshotTime) {
        this.segmentSnapshotTime.set(snapshotTime);
    }

    public void onCompactionConfigUpdated(DruidCompactionConfig compactionConfig) {
        HashSet compactionEnabledDatasources = new HashSet();
        if (compactionConfig.getCompactionConfigs() != null) {
            compactionConfig.getCompactionConfigs().forEach(config -> {
                this.getOrComputeDatasourceStatus(config.getDataSource()).cleanupStaleTaskStatuses();
                compactionEnabledDatasources.add(config.getDataSource());
            });
        }
        HashSet allDatasources = new HashSet(this.datasourceStatuses.keySet());
        allDatasources.forEach(datasource -> {
            if (!compactionEnabledDatasources.contains(datasource)) {
                this.datasourceStatuses.remove(datasource);
            }
        });
    }

    public void onTaskSubmitted(ClientCompactionTaskQuery taskPayload, CompactionCandidate candidateSegments) {
        this.submittedTaskIdToSegments.put(taskPayload.getId(), candidateSegments);
        this.getOrComputeDatasourceStatus(taskPayload.getDataSource()).handleSubmittedTask(candidateSegments);
    }

    public void onTaskFinished(String taskId, TaskStatus taskStatus) {
        if (!taskStatus.isComplete()) {
            return;
        }
        CompactionCandidate candidateSegments = this.submittedTaskIdToSegments.remove(taskId);
        if (candidateSegments == null) {
            return;
        }
        Interval compactionInterval = candidateSegments.getUmbrellaInterval();
        this.getOrComputeDatasourceStatus(candidateSegments.getDataSource()).handleCompletedTask(compactionInterval, taskStatus);
    }

    private DatasourceStatus getOrComputeDatasourceStatus(String datasource) {
        return this.datasourceStatuses.computeIfAbsent(datasource, ds -> new DatasourceStatus());
    }

    private static class DatasourceStatus {
        static final DatasourceStatus EMPTY = new DatasourceStatus();
        final ConcurrentHashMap<Interval, CompactionTaskStatus> intervalToTaskStatus = new ConcurrentHashMap();

        private DatasourceStatus() {
        }

        void handleCompletedTask(Interval compactionInterval, TaskStatus taskStatus) {
            CompactionTaskStatus lastKnownStatus = this.intervalToTaskStatus.get(compactionInterval);
            DateTime now = DateTimes.nowUtc();
            CompactionTaskStatus updatedStatus = taskStatus.isSuccess() ? new CompactionTaskStatus(TaskState.SUCCESS, now, 0) : (lastKnownStatus == null || lastKnownStatus.getState().isSuccess() ? new CompactionTaskStatus(TaskState.FAILED, now, 1) : new CompactionTaskStatus(TaskState.FAILED, now, lastKnownStatus.getNumConsecutiveFailures() + 1));
            this.intervalToTaskStatus.put(compactionInterval, updatedStatus);
        }

        void handleSubmittedTask(CompactionCandidate candidateSegments) {
            Interval interval = candidateSegments.getUmbrellaInterval();
            CompactionTaskStatus lastStatus = this.intervalToTaskStatus.get(interval);
            DateTime now = DateTimes.nowUtc();
            if (lastStatus == null || !lastStatus.getState().isFailure()) {
                this.intervalToTaskStatus.put(interval, new CompactionTaskStatus(TaskState.RUNNING, now, 0));
            } else {
                this.intervalToTaskStatus.put(interval, new CompactionTaskStatus(TaskState.RUNNING, now, lastStatus.getNumConsecutiveFailures()));
            }
        }

        void cleanupStaleTaskStatuses() {
            DateTime now = DateTimes.nowUtc();
            HashSet staleIntervals = new HashSet();
            this.intervalToTaskStatus.forEach((interval, taskStatus) -> {
                if (taskStatus.getUpdatedTime().plus((ReadableDuration)MAX_STATUS_RETAIN_DURATION).isBefore((ReadableInstant)now)) {
                    staleIntervals.add(interval);
                }
            });
            staleIntervals.forEach(this.intervalToTaskStatus::remove);
        }
    }
}

